<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Template;
use App\Models\EmailLog;
use Illuminate\Support\Facades\Mail;
use App\Mail\DbTemplateMail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class EmailCampaignController extends Controller
{
        // Formulaire simple pour uploader CSV et choisir template
    public function showForm()
    {
        $templates = Template::all();
        return view('pages.form', compact('templates'));
    }

    // Upload CSV and store file
    public function uploadCsv(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csv_file' => 'required|file|mimes:csv,txt',
            'template_id' => 'required|exists:templates,id',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $path = $request->file('csv_file')->store('uploads'); // storage/app/uploads/...
        // Save path in session for next step (or database)
        session(['email_csv_path' => $path, 'template_id' => $request->template_id]);

        return redirect()->route('email_campaign.preview');
    }

    // Preview first lines and option to send
    public function preview()
    {
        $path = session('email_csv_path');
        $template_id = session('template_id');

        if (! $path || ! $template_id) {
            return redirect()->route('email_campaign.form')->with('error', 'Aucun CSV ou template sélectionné.');
        }

        $fullPath = storage_path('app/' . $path);
        $rows = [];
        if (($handle = fopen($fullPath, 'r')) !== false) {
            $header = fgetcsv($handle);
            while (($data = fgetcsv($handle)) !== false) {
                $rows[] = array_combine($header, $data);
                if (count($rows) >= 10) break; // preview max 10
            }
            fclose($handle);
        }

        $template = Template::findOrFail($template_id);
        return view('pages.preview', compact('rows','template','path'));
    }

    // Send campaign (synchronous; pour la production utilisez queue())
    public function sendCampaign(Request $request)
    {
        $path = session('email_csv_path');
        $template_id = session('template_id');

        if (! $path || ! $template_id) {
            return redirect()->route('email_campaign.form')->with('error', 'Aucun CSV ou template sélectionné.');
        }

        $template = Template::findOrFail($template_id);
        $fullPath = storage_path('app/' . $path);

        if (! file_exists($fullPath)) {
            return redirect()->route('email_campaign.form')->with('error', 'Fichier CSV introuvable.');
        }

        $sent = 0;
        $errors = [];

        if (($handle = fopen($fullPath, 'r')) !== false) {
            $header = fgetcsv($handle);
            while (($data = fgetcsv($handle)) !== false) {
                $row = array_combine($header, $data);
                $email = $row['email'] ?? null;
                $name = $row['name'] ?? null;
                $phone = $row['phone'] ?? null;

                if (! filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $errors[] = "Email invalide: {$email}";
                    continue;
                }

                // Create log
                $log = EmailLog::create([
                    'email' => $email,
                    'name' => $name,
                    'phone' => $phone,
                    'template_id' => $template->id,
                ]);

                // Prepare body + replace variables
                $body = $template->body;
                $search = ['{{ name }}','{{phone}}','{{ phone }}'];
                $replace = [e($name), e($phone), e($phone)];
                $body = str_replace($search, $replace, $body);

                // Append tracking pixel
                $trackUrl = url("/track-email/{$log->id}.png");
                $body .= '<img src="' . $trackUrl . '" width="1" height="1" style="display:none">';

                try {
                    // For production, use ->queue()
                    Mail::to($email)->send(new DbTemplateMail($template->subject ?? 'Message', $body));
                    $sent++;
                } catch (\Exception $e) {
                    $errors[] = "Erreur envoi {$email}: " . $e->getMessage();
                }
            }
            fclose($handle);
        }

        return view('pages.result', compact('sent','errors'));
    }

    // Stats page
    public function stats()
    {
        $total = EmailLog::count();
        $opened = EmailLog::where('opened', true)->count();
        $byTemplate = EmailLog::selectRaw('template_id, count(*) as total, sum(opened) as opened')
                    ->groupBy('template_id')->get();

        return view('pages.stats', compact('total','opened','byTemplate'));
    }
}
